using System;
using System.Xml;
using System.Text;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// This class represents standard SOAP fault info structure used 
	/// in VistALink fault messages. It contains information about error occured at remote server. 
	/// </summary>
	public class FaultInfo
	{
		// Required parameters
		private readonly FaultCode _faultCode;
		private readonly string _faultString;	

		// Optional parameter		
		private readonly string _faultActor;

		// Optional error details (<error> element)
		private readonly string _errorCode; // required in <error> element
		private readonly string _errorType ;  // optional 
		private readonly string _errorMessage; // required in <error> element

		// Constants used in XML serialization/deserialization		
		private const string XMLCONSTS_FAULT_NODE_NAME = "Fault";
		private const string XMLCONSTS_FAULT_CODE_NODE_NAME = "FaultCode";
		private const string XMLCONSTS_FAULT_STRING_NOGE_NAME = "FaultString";
		private const string XMLCONSTS_FAULT_ACTOR_NODE_NAME = "FaultActor";
		private const string XMLCONSTS_DETAIL_NODE_NAME = "Detail";
		private const string XMLCONSTS_ERROR_NODE_NAME = "Error";
		private const string XMLCONSTS_MESSAGE_NODE_NAME = "Message";
		private const string XMLCONSTS_CODE_MSG_ATTRIBUTE_NAME = "code";
		private const string XMLCONSTS_TYPE_MSG_ATTRIBUTE_NAME = "type";

		/// <summary>
		/// Contstructor specifying fault parameters. 
		/// </summary>
		/// <param name="faultCode">Fault code.</param>
		/// <param name="faultString">Fault string.</param>
		/// <param name="faultActor">Fault actor.</param>
		/// <param name="errorCode">Error code.</param>
		/// <param name="errorType">Error type.</param>
		/// <param name="errorMessage">Error message.</param>
		public FaultInfo( FaultCode faultCode, string faultString, string faultActor, string errorCode, string errorType, string errorMessage ) 
		{
			if( faultCode == null )
				throw( new ArgumentNullException( "faultCode" ) );

			if( faultString == null )
				throw( new ArgumentNullException( "faultString" ) );

			if( faultString == String.Empty ) 
				throw( new ArgumentOutOfRangeException( SR.Exceptions.FaultMessageFaultStringCannotBeEmpty() ) );

			_faultCode = faultCode;
			_faultString = faultString;			
			_faultActor = faultActor;
			_errorCode = errorCode;
			_errorType = errorType;
			_errorMessage = errorMessage;
		} 


		/// <summary>
		/// Deserialization constructor. Creates instance of class from XML, 
		/// previously created by deserialization of object of the same class. 
		/// </summary>
		/// <param name="rootElement">
		///		Root lement is either fault element itself or parent node for fault structure.
		///		This element must have exactly one fault structure among its children.
		///	</param>
		public FaultInfo( XmlElement rootElement )
		{
			if( rootElement == null )
				throw( new ArgumentNullException( "rootElement" ) );

			XmlElement _faultElement;

			if(	rootElement.Name == XMLCONSTS_FAULT_NODE_NAME )
				_faultElement = rootElement;
			else
				_faultElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( rootElement, XMLCONSTS_FAULT_NODE_NAME );

			_faultCode = FaultCode.Parse( XmlUtility.ParseGetRequiredElementByUniqueTagName( _faultElement, XMLCONSTS_FAULT_CODE_NODE_NAME ).InnerText );

			_faultString = XmlUtility.ParseGetRequiredElementByUniqueTagName( _faultElement, XMLCONSTS_FAULT_STRING_NOGE_NAME ).InnerText;

			if( _faultString == null || _faultString == String.Empty )
				throw( new XmlParseException( SR.Exceptions.FaultMessageFaultStringCannotBeEmpty() ) );

			XmlElement _elementBuf = XmlUtility.ParseGetOptionalElementByUniqueTagName( _faultElement, XMLCONSTS_FAULT_ACTOR_NODE_NAME );
			_faultActor = ( _elementBuf != null ) ? _elementBuf.InnerText : null;

			if( (_elementBuf = XmlUtility.ParseGetOptionalElementByUniqueTagName( _faultElement, XMLCONSTS_DETAIL_NODE_NAME ) ) == null )
				return;

			_elementBuf = XmlUtility.ParseGetRequiredElementByUniqueTagName( _elementBuf, XMLCONSTS_ERROR_NODE_NAME );
			_errorCode = XmlUtility.ParseGetRequiredAttributeValue( _elementBuf, XMLCONSTS_CODE_MSG_ATTRIBUTE_NAME );

			if( _elementBuf.HasAttribute( XMLCONSTS_TYPE_MSG_ATTRIBUTE_NAME ) )
				_errorType = _elementBuf.GetAttribute( XMLCONSTS_TYPE_MSG_ATTRIBUTE_NAME );

			_elementBuf = XmlUtility.ParseGetRequiredElementByUniqueTagName( _elementBuf, XMLCONSTS_MESSAGE_NODE_NAME );
			_errorMessage = _elementBuf.InnerText;
		}

		/// <summary>
		/// XML serialization method writing out XML representation of the fault info structure.
		/// </summary>
		/// <param name="writer">XmlWriter to use.</param>
		public void WriteFaultInfoToXml( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_FAULT_NODE_NAME );

			writer.WriteElementString( XMLCONSTS_FAULT_CODE_NODE_NAME, _faultCode.ToString() );
			writer.WriteElementString( XMLCONSTS_FAULT_STRING_NOGE_NAME, _faultString );
				
			if( _faultActor != null )
				writer.WriteElementString( XMLCONSTS_FAULT_ACTOR_NODE_NAME, _faultActor );
				
			if( _errorCode != null ) // error details group is optional, however, error code is required in this group
			{
				writer.WriteStartElement( XMLCONSTS_DETAIL_NODE_NAME );
						
				writer.WriteStartElement( XMLCONSTS_ERROR_NODE_NAME );
				writer.WriteAttributeString( XMLCONSTS_CODE_MSG_ATTRIBUTE_NAME, _errorCode );
				writer.WriteAttributeString( XMLCONSTS_TYPE_MSG_ATTRIBUTE_NAME, _errorType );
									
				XmlUtility.GenWriteElementWithCData( writer, XMLCONSTS_MESSAGE_NODE_NAME, _errorMessage );

				writer.WriteEndElement();

				writer.WriteEndElement();
			}									

			writer.WriteEndElement();
		}

		/// <summary>
		/// Fault code. Uses standard SOAP values. Required.
		/// </summary>
		public FaultCode FaultCode 
		{
			get
			{
				return _faultCode;
			}
		}

		/// <summary>
		/// Fault string briefly describing fault. Required.
		/// </summary>
		public string FaultString
		{
			get
			{
				return _faultString;
			}
		}

		/// <summary>
		/// Fault actor. Optional.
		/// </summary>
		public string FaultActor
		{
			get
			{
				return _faultActor;
			}
		}

		/// <summary>
		/// Fault error code. Optional.
		/// </summary>
		public string ErrorCode
		{
			get
			{
				return _errorCode;
			}
		}

		/// <summary>
		/// Error type. Optional.
		/// </summary>
		public string ErrorType
		{
			get
			{
				return _errorType;
			}
		}

		/// <summary>
		/// Detailed error message. Optional. 
		/// </summary>
		public string ErrorMessage
		{
			get
			{
				return _errorMessage;
			}
		}

		/// <summary>
		/// This simple private class is used to create ToString() 
		/// representation of fault info. 
		/// </summary>
		private class ToStringBuilder
		{
			StringBuilder _sb;

			public ToStringBuilder() 
			{
				_sb = new StringBuilder();
			}

			public void AppendLine( string stringToAdd )
			{
				_sb.Append( stringToAdd	);
				_sb.Append( Environment.NewLine );
			}

			public void AppendValuePairLine( string valueName, string valueToShow )
			{
				_sb.AppendFormat( "{0}: {1}", valueName, ConvertNullStringToNA( valueToShow ) );
				_sb.Append( Environment.NewLine );
			}

			private static string ConvertNullStringToNA( string inputString )
			{
				return inputString == null ||  inputString.Trim() == String.Empty ? "N/A" : inputString;
			}

			public override string ToString()
			{
				return _sb.ToString();
			}
		}

		/// <summary>
		/// Override for standard ToString() method returning developer-friendly
		/// representation of fault info data.
		/// </summary>
		/// <returns></returns>
		public override string ToString()
		{
			ToStringBuilder _tsb = new ToStringBuilder();
			
			_tsb.AppendLine( "Fault information:" );			
			_tsb.AppendValuePairLine( "Code", _faultCode.ToString() );
			_tsb.AppendValuePairLine( "String", _faultString );
			_tsb.AppendValuePairLine( "Actor", _faultActor );
			_tsb.AppendValuePairLine( "Error code", _errorCode );
			_tsb.AppendValuePairLine( "Error type", _errorType );
			_tsb.AppendValuePairLine( "Error message", _errorMessage );

			return _tsb.ToString();
		}

	}
}
